//**************************************************************************/
// Copyright (c) 2010 Autodesk, Inc.
// All rights reserved.
// 
// These coded instructions, statements, and computer programs contain
// unpublished proprietary information written by Autodesk, Inc., and are
// protected by Federal copyright law. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without
// the prior written consent of Autodesk, Inc.
//**************************************************************************/

// width and height
float2 gFinalres : ViewportPixelSize;

// edge color
float4 gEdgeColor = {0.0f, 0.0f, 0.0f, 1.0f};


// true means don't draw any lines for this effect
bool gIgnoreLines = false;

// How much the paper is repeated vertically.
float gPaperScale = 1.5f;

// How much to offset the paper, in UV coordinates. If you are panning the view, you
// may wish to adjust this offset on the fly so that the paper pans with the objects,
// to avoid the shower-door effect. The downside is that you then have to save and track
// this offset forever...
float2 gPaperOffset = float2(0,0);

// How much to vary the noise as we move from pixel to pixel. A higher value here makes the
// noise vary more rapidly
float gNoiseScale = 0.1f;
// How strong the noise is for wobbling the lines
float gLineNoiseAmp = 0.002f;
// How strong the noise is for wobbling the edges of the faces
float gFillNoiseAmp = 0.015f;
float gPencilScale = 10.0f;	

// How much to scale the fill color component
float gSceneMul = 1.0f;
// What to add to the fill color component (after scaling), e.g. a gray
float3 gSceneAdd = float3(0.4f, 0.4f, 0.4f);
// How far away in UV space to look on the texture to get the gradient.
float gGradientWidth =  0.0001f;
// Amplitude of the wobble of the filled area
float gFrayStrength = 0.7f;


// World-view-projection transformation.
float4x4 gWVPXf : WorldViewProjection < string UIWidget = "None"; >;


texture SceneTexture;

sampler2D SceneSampler = sampler_state
{
    Texture = <SceneTexture>;
};

texture LineTexture;

sampler2D LineSampler = sampler_state
{
   Texture = <LineTexture>;
};

texture PaperTexture
<
    string ResourceName = "paper2.jpg";
>;


sampler2D PaperSampler = sampler_state
{
   Texture = <PaperTexture>;
};

texture NoiseTexture
<
    string ResourceName = "noise.bmp";
>;

sampler2D NoiseSampler = sampler_state
{
   Texture = <NoiseTexture>;
};


// Vertex shader input structure.
struct VS_INPUT
{
    float4 Pos : POSITION;
    float2 UV : TEXCOORD0;
};

// Vertex shader output structure.
struct VS_TO_PS
{
    float4 HPos : POSITION;
    float2 UV : TEXCOORD0;
};
float threshold2(float x,float bias, float contrast)
{
    float c = (1-contrast)/2;
    if(c ==0)
        c = 0.000001;
    bias = 1-bias;
    float a = bias-c;
    float b = bias+c;
    
    float ret;

    if (a==b) 
        ret = x;
    else
        ret = (x-a)/(b-a);
        
    return clamp(ret,0,1);
}


float2 CalcGradient(sampler2D samp, float2 texcoord, float width)
{
    //float xy0 = tex2D(samp, texcoord).r;	
    float x1 = tex2D(samp, texcoord + float2(-width, 0)).r;
    float x2 = tex2D(samp, texcoord + float2(width, 0)).r;
    float y1 = tex2D(samp, texcoord + float2(0, -width)).r;
    float y2 = tex2D(samp, texcoord + float2(0, width)).r;
    
    // xy0 cancels out! See Shankel in "Game Programming Gems 3" for a
    // justification, and when this can break down.
    //return float2((xy0-x1) + (x2-xy0),(xy0-y1) + (y2-xy0));
    return float2(x2-x1,y2-y1);
}

// Vertex shader.
VS_TO_PS VS_NPR(VS_INPUT In)
{
    VS_TO_PS Out;
    
    // Transform the position from object space to clip space for output.
    Out.HPos = mul(In.Pos, gWVPXf);
    
    // Pass the texture coordinates unchanged.
    Out.UV = In.UV;
    
    return Out;
}



// Pixel shader.
float4 PS_NPR(VS_TO_PS In) : COLOR0
{
    // compute aspect ratio, to keep paper scaled properly
    float2 aspect;
    aspect.x = gFinalres.x / gFinalres.y;
    aspect.y = 1.0f;
    
    float2 modifiedUV = (In.UV - gPaperOffset) * gPaperScale * aspect;
    // compute the directional wobble direction on the fill due to the paper, i.e. hills
    // and valleys of the paper affecting the fill.
    float2 pdist = CalcGradient(PaperSampler, modifiedUV, gGradientWidth);

    // get a noise value for a secondary wobble.
    float fillNoiseEval = tex2D(NoiseSampler, In.UV * gNoiseScale).r * 2 - 1;

    // Compute a new location to sample the faces, wobbling by the wobble direction.
    // Note that gFillNoiseAmp is indeed a scalar and so there's a diagonal (upper-left
    // to lower-right) bias for this term: turn it way up and it's visible. However,
    // no one ever turns it way up. Higher gives more fill wobble.
    float2 fillNoiseTex = In.UV + pdist * gFrayStrength + fillNoiseEval * gFillNoiseAmp;

    // using this displaced UV, grab the fill color at that location.
    float4 scene = tex2D(SceneSampler, fillNoiseTex);
    
    // Scale fill color found and add gray to the color to wash it out, then clamp.
    // Not a great control, but available.
    scene = saturate(scene * gSceneMul + float4(gSceneAdd, 0.0f));

    // Get the line color, if affected
    float4 lines = float4(1.0f, 1.0f, 1.0f, 1.0f);
    if (!gIgnoreLines)
    {
        // compute line wobble due to noise
        // (note, from a different part of the noise texture than used for fillNoiseEval above).
        float noiseval = tex2D(NoiseSampler, (In.UV + float2(0.5f, 0.5f)) * gNoiseScale).r * 2 - 1;
        float2 noisetex = In.UV + noiseval * gLineNoiseAmp;
        // use this wobbled UV to get the line at this point
        lines = tex2D(LineSampler, noisetex);
    }
        
    // Finally, the lines are faded by the paper's texture
    
    // Add the scene and paper colors together and subtract 1.
    // get paper color
    float4 paper = float4(tex2D(PaperSampler, modifiedUV).rgb, 1.0f);
    // add to scene, subtract 1 to bring it into a useful range.
    scene = paper + scene - 1;
    
    // get yet another noise value, this time scaled by pencil scale
    float pencilNoise = tex2D(NoiseSampler, In.UV*gPencilScale).r*2.0f-1.0f;
    // ensure the value is between 0.9 and 0
    pencilNoise = threshold2(pencilNoise,0.9,0);
    // blend between scene and pencilNoise * scene using line's red channel.
    // Lines are assumed to come in grayscale and be affected by color later.
    // When the line is retrieved from LineSampler, the red value in that texture
    // is how much the line does not covers the pixel (e.g. 0.0 means fully covered).
    // Basically, this makes only the lines be affected a little by the pencilNoise.
    // So, first term is how much of the fill (scene) is used, second how much the line
    // is used.
    float sceneStrength = lines.r + (1.0f - lines.r) * pencilNoise;
    float4 output = scene * sceneStrength + gEdgeColor * (1.0f - sceneStrength);
    
    return output;	
}

// The main technique.
technique Main
{
    pass p0
    {
#ifdef SHADER_MODEL_3
        VertexShader = compile vs_3_0 VS_NPR();
        PixelShader = compile ps_3_0 PS_NPR();
#else
        VertexShader = compile vs_2_0 VS_NPR();
        PixelShader = compile ps_2_0 PS_NPR();
#endif
    }
}

